// Theme loader for Sparx Maths
(function() {
    'use strict';

    const isSparxDomain = window.location.hostname.includes('sparx');
    if (!isSparxDomain) return;

    console.log('Sparx Themes Loader activated');

    let currentTheme = 'default';
    let textReplacementInterval = null;
    let mutationObserver = null;

    // Apply saved theme when page loads
    function applySavedTheme() {
        chrome.storage.sync.get('sparxTheme', function(data) {
            currentTheme = data.sparxTheme || 'default';
            applyTheme(currentTheme);
        });
    }

    function applyTheme(themeName) {
        console.log('Applying theme:', themeName);
        
        // Remove existing theme
        const existingTheme = document.getElementById('sparx-theme');
        if (existingTheme) {
            existingTheme.remove();
        }

        // Clear any existing text replacement intervals
        if (textReplacementInterval) {
            clearInterval(textReplacementInterval);
            textReplacementInterval = null;
        }

        // Stop mutation observer
        if (mutationObserver) {
            mutationObserver.disconnect();
            mutationObserver = null;
        }

        if (themeName === 'default') {
            // Completely reset to original Sparx styling
            revertTextReplacements();
            return;
        }

        // Load theme CSS
        const link = document.createElement('link');
        link.id = 'sparx-theme';
        link.rel = 'stylesheet';
        link.href = chrome.runtime.getURL(`themes/${themeName}-theme.css`);
        document.head.appendChild(link);

        console.log(`Applied ${themeName} theme`);

        // Apply theme-specific modifications
        if (themeName === 'hacker') {
            startHackerModifications();
        }
    }

    function startHackerModifications() {
        // Apply immediately
        applyHackerTextReplacements();
        
        // Set up continuous replacement for dynamic content
        textReplacementInterval = setInterval(applyHackerTextReplacements, 1000);
        
        // Also watch for DOM changes
        mutationObserver = new MutationObserver(function(mutations) {
            let shouldReplace = false;
            mutations.forEach(function(mutation) {
                if (mutation.addedNodes.length > 0) {
                    shouldReplace = true;
                }
            });
            if (shouldReplace) {
                setTimeout(applyHackerTextReplacements, 100);
            }
        });
        
        mutationObserver.observe(document.body, {
            childList: true,
            subtree: true
        });
    }

    function applyHackerTextReplacements() {
        // Replace welcome message
        replaceTextInDocument(/Hey\s+([^,!\n]+)[,!\n]?/gi, 'Hey, $1 The Hacker!');
        
        // Replace homework description
        replaceTextInDocument(/answer every question correctly/gi, 'start hacking every question correctly');
        
        // Replace other Sparx text
        replaceTextInDocument(/Sparx Maths/gi, 'Terminal Maths');
        replaceTextInDocument(/complete your homework/gi, 'complete your mission');
    }

    function revertTextReplacements() {
        // Revert hacker text back to original
        replaceTextInDocument(/Hey,\s+([^!]+) The Hacker!/gi, 'Hey, $1!');
        replaceTextInDocument(/start hacking every question correctly/gi, 'answer every question correctly');
        replaceTextInDocument(/Terminal Maths/gi, 'Sparx Maths');
        replaceTextInDocument(/complete your mission/gi, 'complete your homework');
    }

    function replaceTextInDocument(pattern, replacement) {
        try {
            // Use TreeWalker to safely replace text
            const walker = document.createTreeWalker(
                document.body,
                NodeFilter.SHOW_TEXT,
                {
                    acceptNode: function(node) {
                        // Only process nodes that contain text and aren't in scripts/styles
                        if (node.parentNode.nodeName === 'SCRIPT' || 
                            node.parentNode.nodeName === 'STYLE' ||
                            node.parentNode.nodeName === 'NOSCRIPT') {
                            return NodeFilter.FILTER_REJECT;
                        }
                        if (node.textContent.match(pattern)) {
                            return NodeFilter.FILTER_ACCEPT;
                        }
                        return NodeFilter.FILTER_SKIP;
                    }
                },
                false
            );

            let node;
            const nodesToReplace = [];
            
            // Collect nodes first to avoid modifying during iteration
            while (node = walker.nextNode()) {
                nodesToReplace.push(node);
            }
            
            // Replace text in collected nodes
            nodesToReplace.forEach(node => {
                node.textContent = node.textContent.replace(pattern, replacement);
            });
            
        } catch (error) {
            console.log('Text replacement safe error:', error);
        }
    }

    // Initialize
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', function() {
            setTimeout(applySavedTheme, 100);
        });
    } else {
        setTimeout(applySavedTheme, 100);
    }

    // Listen for theme changes
    chrome.storage.onChanged.addListener(function(changes) {
        if (changes.sparxTheme) {
            applyTheme(changes.sparxTheme.newValue);
        }
    });

    chrome.runtime.onMessage.addListener(function(request, sender, sendResponse) {
        if (request.action === 'applyTheme') {
            applyTheme(request.theme);
            sendResponse({success: true});
        }
        return true;
    });

})();